<?php
/**
 * 数据分析功能类
 *
 * Data analysis function class
 */
class das
{

    /**
     * 构造方法，初始化自动加载类方法
     *
     * Constructor, initializing the method of auto loading class.
     *
     * @access public
     */
    public function __construct()
    {
        spl_autoload_register(array($this, 'autoLoad'), true, true);
        define('LIB_ROOT', dirname(dirname(__DIR__)) . DS . 'lib');
    }

    /**
     * 自动加载类文件
     *
     * Load class file.
     * @param string  $class
     *
     * @access public
     */
    private function autoLoad($class)
    {
        $fileName = strtolower(substr($class, 0, strlen($class) - 6));
        $dir      = $this->methodDir($fileName);
        $fileName = $dir === false ? $fileName : $dir . DS . $fileName;
        $method   = __DIR__ . DS . 'methods' . DS . $fileName . '.php';
        $simulate = __DIR__ . DS . 'simulate' . DS . strtolower(substr($class, 0, strlen($class) - 8)) . '.php';
        if(file_exists($method)) include_once($method);
        if(file_exists($simulate)) include_once($simulate);
    }

    /**
     * 调用分析方法
     *
     * Invoke analysis method
     *
     * @param string     $methodName    The analysis method name
     * @param dataframe  $dataframe     DataFrame object
     * @param object     $settings      The analysis method settings object
     * @static
     * @access public
     * @return array
     */
    public function invokeAnalysisMethod($methodName, $dataframe, $settings)
    {
        $className = $methodName . 'Method';
        $method    = new $className();

        $results = $method->func($dataframe, $settings);

        return $results;
    }

    /**
     * 调用模拟数据方法
     *
     * Invoke simulate data.
     *
     * @param string     $name          The simulate method name
     * @param object     $settings      The simulate method settings object
     * @static
     * @access public
     * @return array
     */
    public function invokeSimulateData($name, $dataframe, $settings)
    {
        $className = $name . 'Simulate';
        $method    = new $className();
        return $method->data($dataframe, $settings);
    }

    /**
     * 加载语言文件
     *
     * Load lang file.
     *
     * @param string     $langName  zh-cn|en
     * @access public
     */
    public function initDasLang($langName)
    {
        include(__DIR__ . DS . 'lang' . DS . $langName . '.php');
        global $dasLang;
        $dasLang = $lang;
    }

    /**
     * 获取方法设置
     *
     * Get method setting.
     *
     * @param string  $methodName
     * @param string  $langName    zh-cn|en
     * @static
     * @access public
     * @return array
     */
    public function getMethodSettings($methodName, $langName)
    {
        $className = $methodName . 'Method';
        $method    = new $className();

        $this->initDasLang($langName);
        return $method->getSettings();
    }

    /**
     * 获取模拟方法设置
     *
     * Get simulate method setting.
     *
     * @param string     $methodName
     * @param string     $langName    zh-cn|en
     * @param object     $dataframe
     * @static
     * @access public
     * @return array
     */
    public function getSimulateSettings($methodName, $langName, $dataframe = null)
    {
        $className = $methodName . 'Simulate';
        $method    = new $className();

        $this->initDasLang($langName);
        return $method->getSettings($dataframe);
    }

    /**
     * 设定图形的数据
     * Sets the data for the drawing.
     *
     * @param string  $methodName
     * @access public
     * @return array
     */
    public function getDiagramConfig($methodName, $langName)
    {
        $className = $methodName . 'Method';
        $method    = new $className();

        $this->initDasLang($langName);
        return $method->getConfig();
    }

    private function methodDir($methodName)
    {
        $config = array();
        $config['basicstatistic']     = 'statistic';
        $config['graphsummary']       = 'statistic';
        $config['montecarlo']         = 'statistic';
        $config['processgroup']       = 'statistic';
        $config['normaldistribution'] = 'statistic';
        $config['poisson']            = 'statistic';

        $config['chisquare']   = 'hypothesis';
        $config['eqvariance']  = 'hypothesis';
        $config['normaltest']  = 'hypothesis';
        $config['onesample']   = 'hypothesis';
        $config['onevariance'] = 'hypothesis';
        $config['onewayanova'] = 'hypothesis';
        $config['pairt']       = 'hypothesis';
        $config['proportion1'] = 'hypothesis';
        $config['proportion2'] = 'hypothesis';
        $config['twoanova']    = 'hypothesis';
        $config['twosample']   = 'hypothesis';

        $config['c']     = 'control';
        $config['imr']   = 'control';
        $config['spc']   = 'control';
        $config['u']     = 'control';
        $config['xbarr'] = 'control';
        $config['xbars'] = 'control';

        $config['barplot']             = 'graphic';
        $config['boxplot']             = 'graphic';
        $config['bubbleplot']          = 'graphic';
        $config['empiricalplot']       = 'graphic';
        $config['individualvalueplot'] = 'graphic';
        $config['lineplot']            = 'graphic';
        $config['pieplot']             = 'graphic';
        $config['radarplot']           = 'graphic';
        $config['scatterplot']         = 'graphic';

        $config['correlation']          = 'analysis';
        $config['generallinearmodel']   = 'analysis';
        $config['gompertz']             = 'analysis';
        $config['linearregression']     = 'analysis';
        $config['multipleregression']   = 'analysis';
        $config['polynomialregression'] = 'analysis';

        return isset($config[$methodName]) ? $config[$methodName] : false;
    }
}
