<?php

namespace MathPHP\Tests\Functions\Map;

use MathPHP\Functions\Map\Multi;
use MathPHP\Exception;

class MultiTest extends \PHPUnit\Framework\TestCase
{

    /**
     * @test         add two arrays
     * @dataProvider dataProviderForAddTwoArrays
     * @param        array $xs
     * @param        array $ys
     * @param        array $expected
     * @throws       \Exception
     */
    public function testAddTwoArrays($xs, $ys, $expected)
    {
        // When
        $sums = Multi::add($xs, $ys);

        // Then
        $this->assertEquals($expected, $sums);
    }

    public function dataProviderForAddTwoArrays()
    {
        return [
            [
                [1, 2, 3, 4],
                [2, 3, 4, 5],
                [3, 5, 7, 9],
            ],
            [
                [1, 2, 3, 4],
                [6, 6, 6, 6],
                [7, 8, 9, 10],
            ],
        ];
    }

    /**
     * @test         add multiple arrays
     * @dataProvider dataProviderForAddMulti
     * @param        array $expected
     * @param        array ...$arrays
     * @throws       \Exception
     */
    public function testAddMulti($expected, ...$arrays)
    {
        // When
        $sums = Multi::add(...$arrays);

        // Then
        $this->assertEquals($expected, $sums);
    }

    public function dataProviderForAddMulti()
    {
        return [
            [
                [3, 5, 7, 9],
                [1, 2, 3, 4],
                [2, 3, 4, 5],
            ],
            [
                [7, 8, 9, 10],
                [1, 2, 3, 4],
                [6, 6, 6, 6],
            ],
            [
                [6, 7, 9, 10],
                [1, 2, 3, 4],
                [2, 2, 2, 2],
                [3, 3, 4, 4],
            ]
        ];
    }

    /**
     * @test         subtract two arrays
     * @dataProvider dataProviderForSubtractTwoArrays
     * @param        array $xs
     * @param        array $ys
     * @param        array $expected
     * @throws       \Exception
     */
    public function testSubtractTwoArrays($xs, $ys, $expected)
    {
        // When
        $differences = Multi::subtract($xs, $ys);

        // Then
        $this->assertEquals($expected, $differences);
    }

    public function dataProviderForSubtractTwoArrays()
    {
        return [
            [
                [1, 2, 3, 4],
                [2, 3, 4, 5],
                [-1, -1, -1, -1],
            ],
            [
                [1, 2, 3, 4],
                [6, 6, 6, 6],
                [-5, -4, -3, -2],
            ],
        ];
    }

    /**
     * @test         subtract multiple arrays
     * @dataProvider dataProviderForSubtractMulti
     * @param        array $expected
     * @param        array[] $arrays
     * @throws       \Exception
     */
    public function testSubtractMulti($expected, ...$arrays)
    {
        // When
        $differences = Multi::subtract(...$arrays);

        // Then
        $this->assertEquals($expected, $differences);
    }

    public function dataProviderForSubtractMulti()
    {
        return [
            [
                [-1, -1, -1, -1],
                [1, 2, 3, 4],
                [2, 3, 4, 5],
            ],
            [
                [-5, -4, -3, -2],
                [1, 2, 3, 4],
                [6, 6, 6, 6],
            ],
            [
                [3, 3, 4, 4],
                [6, 7, 9, 10],
                [1, 2, 3, 4],
                [2, 2, 2, 2],
            ]
        ];
    }

    /**
     * @test         multiply two arrays
     * @dataProvider dataProviderForMultiplyTwoArrays
     */
    public function testMultiplyTwoArrays($xs, $ys, $expected)
    {
        // When
        $products = Multi::multiply($xs, $ys);

        // Then
        $this->assertEquals($expected, $products);
    }

    public function dataProviderForMultiplyTwoArrays()
    {
        return [
            [
                [1, 2, 3, 4],
                [2, 3, 4, 5],
                [2, 6, 12, 20],
            ],
            [
                [1, 2, 3, 4],
                [6, 6, 6, 6],
                [6, 12, 18, 24],
            ],
        ];
    }

    /**
     * @test         multiply multiple arrays
     * @dataProvider dataProviderForMultiplyMulti
     */
    public function testMultiplyMulti($expected, ...$arrays)
    {
        // When
        $products = Multi::multiply(...$arrays);

        // Then
        $this->assertEquals($expected, $products);
    }

    public function dataProviderForMultiplyMulti()
    {
        return [
            [
                [2, 6, 12, 20],
                [1, 2, 3, 4],
                [2, 3, 4, 5],
            ],
            [
                [6, 12, 18, 24],
                [1, 2, 3, 4],
                [6, 6, 6, 6],
            ],
            [
                [12, 28, 54, 80],
                [6, 7, 9, 10],
                [1, 2, 3, 4],
                [2, 2, 2, 2],
            ]
        ];
    }

    /**
     * @test         divide two arrays
     * @dataProvider dataProviderForDivideTwoArrays
     * @param        array $xs
     * @param        array $ys
     * @param        array $expected
     * @throws       \Exception
     */
    public function testDivideTwoArrays($xs, $ys, $expected)
    {
        // When
        $quotients = Multi::divide($xs, $ys);

        // Then
        $this->assertEquals($expected, $quotients);
    }

    public function dataProviderForDivideTwoArrays()
    {
        return [
            [
                [5, 10, 15, 20],
                [5, 5, 5, 5],
                [1, 2, 3, 4],
            ],
            [
                [5, 10, 15, 20],
                [2.5, 20, 3, 4],
                [2, 0.5, 5, 5],
            ],
        ];
    }

    /**
     * @test         divide multiple arrays
     * @dataProvider dataProviderForDivideMulti
     * @param        array $expected
     * @param        array[] $arrays
     * @throws       \Exception
     */
    public function testDivideMulti($expected, ...$arrays)
    {
        // When
        $quotients = Multi::divide(...$arrays);

        // Then
        $this->assertEquals($expected, $quotients);
    }

    public function dataProviderForDivideMulti()
    {
        return [
            [
                [1, 2, 3, 4],
                [5, 10, 15, 20],
                [5, 5, 5, 5],
            ],
            [
                [2, 0.5, 5, 5],
                [5, 10, 15, 20],
                [2.5, 20, 3, 4],
            ],
            [
                [5, 20, 1, 8],
                [100, 80, 25, 64],
                [10, 2, 5, 2],
                [2, 2, 5, 4],
            ]
        ];
    }

    /**
     * @test         max two arrays
     * @dataProvider dataProviderForMaxTwoArrays
     * @param        array $xs
     * @param        array $ys
     * @param        array $expected
     * @throws       \Exception
     */
    public function testMaxTwoArrays($xs, $ys, $expected)
    {
        // When
        $maxes = Multi::max($xs, $ys);

        // Then
        $this->assertEquals($expected, $maxes);
    }

    public function dataProviderForMaxTwoArrays()
    {
        return [
            [
                [1, 5, 3, 6],
                [5, 5, 5, 5],
                [5, 5, 5, 6],
            ],
            [
                [5, 10, 15, 20],
                [2.5, 20, 3, 4],
                [5, 20, 15, 20],
            ],
        ];
    }

    /**
     * @test         max multiple arrays
     * @dataProvider dataProviderForMaxMulti
     * @param        array $expected
     * @param        array[] $arrays
     * @throws       \Exception
     */
    public function testMaxMulti($expected, ...$arrays)
    {
        // When
        $maxes = Multi::max(...$arrays);

        // Then
        $this->assertEquals($expected, $maxes);
    }

    public function dataProviderForMaxMulti()
    {
        return [
            [
                [5, 10, 15, 20],
                [5, 10, 15, 20],
                [5, 5, 5, 5],
            ],
            [
                [5, 20, 15, 20],
                [5, 10, 15, 20],
                [2.5, 20, 3, 4],
            ],
            [
                [100, 80, 55, 664],
                [100, 80, 25, 64],
                [10, 2, 55, 2],
                [2, 2, 5, 664],
            ]
        ];
    }

    /**
     * @test         min
     * @dataProvider dataProviderForMin
     * @param        array $xs
     * @param        array $ys
     * @param        array $expected
     * @throws       \Exception
     */
    public function testMin($xs, $ys, $expected)
    {
        // When
        $mins = Multi::min($xs, $ys);

        // Then
        $this->assertEquals($expected, $mins);
    }

    public function dataProviderForMin()
    {
        return [
            [
                [1, 5, 3, 6],
                [5, 5, 5, 5],
                [1, 5, 3, 5],
            ],
            [
                [5, 10, 15, 20],
                [2.5, 20, 3, 4],
                [2.5, 10, 3, 4],
            ],
        ];
    }

    /**
     * @test         min multiple arrays
     * @dataProvider dataProviderForMinMulti
     * @param        array $expected
     * @param        array[] $arrays
     * @throws       \Exception
     */
    public function testMinMulti($expected, ...$arrays)
    {
        // When
        $mins = Multi::min(...$arrays);

        // Then
        $this->assertEquals($expected, $mins);
    }

    public function dataProviderForMinMulti()
    {
        return [
            [
                [5, 5, 5, 5],
                [5, 10, 15, 20],
                [5, 5, 5, 5],
            ],
            [
                [2.5, 10, 3, 4],
                [5, 10, 15, 20],
                [2.5, 20, 3, 4],
            ],
            [
                [2, 2, 5, 2],
                [100, 80, 25, 64],
                [10, 2, 55, 2],
                [2, 2, 5, 664],
            ]
        ];
    }

    /**
     * @test   array lengths are not equal exception
     * @throws \Exception
     */
    public function testCheckArrayLengthsException()
    {
        // Given
        $xs = [1, 2, 3];
        $ys = [1, 2];

        // Then
        $this->expectException(Exception\BadDataException::class);

        // When
        Multi::add($xs, $ys);
    }

    /**
     * @test   Only one array exception
     * @throws \Exception
     */
    public function testCheckArrayLengthsExceptionOnlyOneArray()
    {
        // Given
        $xs = [1, 2];

        // Then
        $this->expectException(Exception\BadDataException::class);

        // When
        Multi::add($xs);
    }
}
