<?php
/**
 * The model file of objective module of ZenTaoPMS.
 * @copyright   Copyright 2009-2023 禅道软件（青岛）有限公司(ZenTao Software (Qingdao) Co., Ltd. www.zentao.net)
 * @license     ZPL(https://zpl.pub/page/zplv12.html) or AGPL(https://www.gnu.org/licenses/agpl-3.0.en.html)
 * @author      Yuting Wang <wangyuting@easycorp.ltd>
 * @package     objective
 * @link        https://www.zentao.net
 */
class objectiveModel extends model
{
    /**
     * 获取目标列表。
     * Get objective list.
     *
     * @param  int         $piID
     * @param  string      $included
     * @param  string      $excluded
     * @param  string      $browseType
     * @param  string      $orderBy
     * @param  object      $pager
     * @access public
     * @return array|false
     */
    public function getList($piID, $included = '', $excluded = '', $browseType = 'all', $orderBy = 'id_desc', $pager = null)
    {
        return $this->dao->select('*')->from(TABLE_OBJECTIVE)
            ->where('deleted')->eq('0')
            ->andWhere('pi')->eq($piID)
            ->beginIF($browseType != 'all')->andWhere('type')->eq($browseType)->fi()
            ->beginIF($included)->andWhere('id')->in($included)->fi()
            ->beginIF($excluded)->andWhere('id')->notin($excluded)->fi()
            ->orderBy($orderBy)
            ->page($pager)
            ->fetchAll('id');
    }

    /**
     * 创建一个目标。
     * Create a objective.
     *
     * @param  object    $objective
     * @access public
     * @return int|false
     */
    public function create($objective)
    {
        $this->dao->insert(TABLE_OBJECTIVE)->data($objective)
            ->autoCheck()
            ->batchCheck($this->config->objective->create->requiredFields, 'notempty')
            ->check('name', 'unique')
            ->exec();

        if(dao::isError()) return false;

        /* 保存历史记录。 */
        $objectiveID = $this->dao->lastInsertID();
        $this->loadModel('action')->create('objective', $objectiveID, 'Opened');

        return $objectiveID;
    }

    /**
     * 更新一个目标。
     * Update a objective.
     *
     * @param  object    $objective
     * @access public
     * @return bool
     */
    public function update($objective): bool
    {
        if(empty($objective->id)) return false;
        $oldObjective = $this->fetchByID($objective->id);

        $this->dao->update(TABLE_OBJECTIVE)->data($objective)
            ->autoCheck()
            ->batchCheck($this->config->objective->create->requiredFields, 'notempty')
            ->check('name', 'unique', "id != $objective->id")
            ->where('id')->eq($objective->id)
            ->exec();

        if(dao::isError()) return false;

        /* 保存历史记录。 */
        $changes  = common::createChanges($oldObjective, $objective);
        if($changes)
        {
            $actionID = $this->loadModel('action')->create('objective', $objective->id, 'Edited');
            $this->action->logHistory($actionID, $changes);
        }

        return dao::isError();
    }

    /**
     * 关联团队目标。
     * Link objectives.
     *
     * @param  int    $id
     * @param  string $objectives
     * @access public
     * @return bool
     */
    public function link($id, $objectives)
    {
        $links = $this->dao->select('links')->from(TABLE_OBJECTIVE)->where('id')->eq($id)->fetch('links');
        $links .= ",$objectives";
        $links = trim($links, ',');

        $this->dao->update(TABLE_OBJECTIVE)->set('links')->eq($links)->where('id')->eq($id)->exec();
        return !dao::isError();
    }

    /**
     * 取消关联团队目标。
     * Unlink objectives.
     *
     * @param  int    $id
     * @param  int    $unlinkID
     * @access public
     * @return bool
     */
    public function unlink($id, $unlinkID)
    {
        $links = $this->dao->select('links')->from(TABLE_OBJECTIVE)->where('id')->eq($id)->fetch('links');
        $links = str_replace(",$unlinkID,", ',', ",$links,");
        $links = trim($links, ',');

        $this->dao->update(TABLE_OBJECTIVE)->set('links')->eq($links)->where('id')->eq($id)->exec();
        return !dao::isError();
    }


    /**
     * 检查按钮是否允许被点击。
     * Check clickable.
     *
     * @param  object $object
     * @param  string $action
     * @param  string $module
     * @static
     * @access public
     * @return bool
     */
    public static function isClickable($object, $action): bool
    {
        $action = strtolower($action);

        if($object->deleted) return false;

        if($action == 'link')   return $object->type == 'parent';
        if($action == 'unlink') return $object->type == 'child';
        if($action == 'delete') return $object->type == 'parent';

        return true;
    }
}
