<?php
class imConference extends model
{
    /**
     * Generate number of a conference based on its id.
     * The first character represents the length of id (which is on the tail of the number).
     * A padding of date is inserted between the length of id and the id if the number is not long enough.
     *
     * @param  object $conference
     * @access public
     * @return string
     */
    public function getNumber($conference)
    {
        if (!empty($conference->number)) return $conference->number;

        $idLength = strlen((string)$conference->id);
        if($idLength > 8)
        {
            $number = $idLength . $conference->id;
        }
        else
        {
            /* Pad the number to length of 9 with openedDate if id is not long enough. */
            $dateNum = str_replace('-', '', current(explode(' ', $conference->openedDate)));
            $number = $idLength . substr($dateNum, 0, 8 - $idLength) . $conference->id;
        }

        return $number;
    }

    /**
     * Set the number of a conference.
     *
     * @param  int    $id conference id
     * @param  string $number conference number
     * @access public
     * @return bool
     */
    public function setNumber($id, $number)
    {
        $this->dao->update(TABLE_IM_CONFERENCE)->set('number')->eq($number)->where('id')->eq($id)->exec();
        return !dao::isError();
    }

    /**
     * Extract id from number.
     *
     * @param  string $number
     * @access public
     * @return int
     */
    public function extractIdFromNumber($number)
    {
        $len = strlen($number);

        /* Use first chars as length of id. */
        if($len > 10)
        {
            $id = substr($number, 2);
        }
        else
        {
            $idLength = intval(substr($number, 0, 1));
            $id = substr($number, $len - $idLength, $idLength);
        }

        return intval($id);
    }

    /**
     * Get a conference by number.
     * The number is generated by the getNumber method.
     *
     * @param  int    $number
     * @access public
     * @return object
     */
    public function getByNumber($number)
    {
        $id = $this->extractIdFromNumber($number);

        return $this->getById($id);
    }

    /**
     * Get the old conference by the id of chat.
     * @deprecated
     * @param  string $chatID
     * @access public
     * @return object
     */
    public function getByChatID($chatID)
    {
        return $this->dao->select('*')->from(TABLE_IM_CONFERENCE)
                         ->where('number')->eq('')
                         ->andWhere('cgid')->eq($chatID)
                         ->orderBy('id_asc')->fetch();
    }

    /**
     * Get the latest conference by the id of chat.
     *
     * @param  string $cgid
     * @access public
     * @return object
     */
    public function getLatestByChatID($cgid)
    {
        return $this->dao->select('*')->from(TABLE_IM_CONFERENCE)
                         ->where('cgid')->eq($cgid)
                         ->orderBy('id_desc')->fetch();
    }

    /**
     * Get conferences by user id.
     *
     * @param  int    $userID
     * @access public
     * @return array
     */
    public function getByUserID($userID)
    {
        $conferences = $this->dao->select('*')->from(TABLE_IM_CONFERENCE)
                                 ->where('status')->eq('open')
                                 ->fetchAll();
        $userConferences = array();
        foreach($conferences as $conference)
        {
            $participants = explode(',', $conference->participants);
            $participants = array_filter($participants);
            if(in_array($userID, $participants)) $userConferences[] = $conference;
        }
        return $userConferences;
    }

    /**
     * Get a conference by id.
     *
     * @param  int    $id
     * @access public
     * @return object
     */
    public function getById($id)
    {
        return $this->dao->select('*')->from(TABLE_IM_CONFERENCE)->where('id')->eq($id)->fetch();
    }

    /**
     * get not started scheduled conferences.
     * @return object
     */
    public function getNotStartedSchedule()
    {
        return $this->dao->select('*')->from(TABLE_IM_CONFERENCE)
                         ->where('status')->eq('notStarted')
                         /* notStarted conferences notify count will be 1 when create. */
                         ->andWhere('sentNotify')->eq(1)
                         ->fetchAll();
    }

    /**
     * get will start scheduled conferences.
     * @return object
     */
    public function getWillStartSchedule()
    {
        return $this->dao->select('*')->from(TABLE_IM_CONFERENCE)
                         ->where('status')->eq('notStarted')
                         /* will start conference notify count should be 2 */
                         ->andWhere('sentNotify')->in(2)
                         ->fetchAll();
    }

    /**
     * get expired detached conference.
     * expire definition: 30 minutes after last action.
     * @return array conference
     */
    public function getExpired()
    {
        $conferences = $this->dao->select('*')->from(TABLE_IM_CONFERENCE)
                         ->where('status')->eq('open')
                         ->andWhere('number')->ne('')
                         ->fetchAll();
        $openedConferences = array();
        foreach($conferences as $conference)
        {
            $lastAction = $this->dao->select('*')->from(TABLE_IM_CONFERENCEACTION)
                                    ->where('rid')->eq($conference->rid)
                                    ->orderBy('id_desc')
                                    ->fetch();
            try
            {
                $lastActionTime = new DateTime($lastAction->date);
            }
            catch (Exception $e)
            {
                continue;
            }
            if (new DateTime('now') > date_add($lastActionTime, DateInterval::createFromDateString('30 minutes')))
            {
                $openedConferences []= $conference;
            }
        }
        return $openedConferences;
    }

    /**
     * Get conference by condition.
     * @param  string $condition
     * @param  array  $options
     * @param  int      $userID
     * @return array
     */
    public function getByCondition($condition, $options, $pager, $userID)
    {
        if(!is_object($options) && is_array($options)) $options = (object)$options;
        $isCreateByMe = property_exists($options, 'isCreateByMe') ? $options->isCreateByMe : false;

        $conferences = array();
        switch($condition)
        {
            case 'onGoing':
                $chatList = $this->loadModel('im')->chatGetGidListByUserID($userID);
                $conferences = $this->dao->select('*')->from(TABLE_IM_CONFERENCE)
                                         ->where('status')->eq('open')
                                         ->andWhere('number')->ne('')
                                         ->beginIF($isCreateByMe)->andWhere('openedBy')->eq($userID)
                                         ->fi()
                                         ->andWhere('participants', true)->like("%,$userID,%")
                                             ->orWhere('invitee')->like("%,$userID,%")
                                             ->orWhere('subscribers')->like("%,$userID,%")
                                             ->orWhere('cgid')->in($chatList)
                                             ->markRight(1)
                                         ->fetchAll();
            break;
            case 'scheduled':
                $conferences = $this->dao->select('*')->from(TABLE_IM_CONFERENCE)
                                         ->where('status')->eq('notStarted')
                                         ->andWhere('type')->eq('scheduled')
                                         ->beginIF($isCreateByMe)->andWhere('openedBy')->eq($userID)
                                         ->fi()
                                         ->andWhere('invitee', true)->like("%,$userID,%")
                                            ->orWhere('subscribers')->like("%,$userID,%")
                                            ->markRight(1)
                                         ->orderBy('id_desc')
                                         ->page($pager)
                                         ->fetchAll();
            break;
            case 'closed':
                $chatList = $this->loadModel('im')->chatGetGidListByUserID($userID);
                $conferences = $this->dao->select('t1.*, t2.hide, t2.user')->from(TABLE_IM_CONFERENCE)->alias('t1')
                                        ->leftJoin(TABLE_IM_CONFERENCEUSER)->alias('t2')
                                        ->on('t2.conference=t1.id and t2.user')->eq($userID)
                                        ->where('(t2.user is null')
                                        ->orWhere('t2.user')->eq($userID)
                                        ->andWhere('t2.hide')->ne(1)
                                        ->markRight(1)
                                        ->andWhere('number')->ne('')
                                        ->andWhere('t1.status')->eq('closed')
                                        ->andWhere('t1.subscribers')->like("%,$userID,%")
                                        ->beginIF($isCreateByMe)->andWhere('t1.openedBy')->eq($userID)
                                        ->fi()
                                        ->andWhere('t1.invitee', true)->like("%,$userID,%")
                                            ->orWhere('t1.subscribers')->like("%,$userID,%")
                                            ->orWhere('t1.cgid')->in($chatList)
                                            ->markRight(1)
                                        ->orderBy('id_desc')
                                        ->page($pager)
                                        ->fetchAll();
            break;
        }
        $thisClass = $this;
        return array_map(function($conference) use ($thisClass) { return $thisClass->format($conference); }, $conferences);
    }

    /**
     * Format conference data.
     *
     * @param  object $conference
     * @access public
     * @return object
     */
    public function format($conference)
    {
        if(empty($conference)) return $conference;

        foreach(array('openedDate', 'startTime', 'endTime') as $key) $conference->$key = $conference->$key == null ? 0 : strtotime($conference->$key);

        if(!empty($conference->number) && !empty($conference->rid) && empty($conference->room))
        {
            $conference->room = $conference->rid;
            unset($conference->rid);
        }

        return $conference;
    }

    /**
     * Create a detached conference.
     *
     * @param  string         $chatID
     * @param  string         $invitee
     * @param  string         $topic
     * @param  string         $type
     * @param  string         $password
     * @param  string         $startTime
     * @param  string         $endTime
     * @param  int            $reminderTime
     * @param  string         $note
     * @param  int            $userID
     * @access public
     * @return object|boolean
     */
    public function createDetached($chatID, $invitee, $type = 'default', $topic = '', $password = '', $startTime = null, $endTime = null, $reminderTime = 0, $note = '', $userID = 0)
    {
        $conferenceData = new stdClass();
        $conferenceData->openedDate = helper::now();

        $this->dao->insert(TABLE_IM_CONFERENCE)->data($conferenceData)->exec();
        if(dao::isError()) return false;

        $id = $this->dao->lastInsertID();
        if(empty($id)) return false;

        $conference = $this->getById($id);
        if(empty($conference)) return false;

        $number = $this->getNumber($conference);
        $this->setNumber($id, $number);

        $conferenceData = new stdClass();
        $conferenceData->cgid         = $chatID;
        $conferenceData->rid          = $number;
        $conferenceData->status       = $type == 'default' ? 'open' : 'notStarted';
        $conferenceData->openedBy     = (int)$userID;
        $conferenceData->openedDate   = helper::now();
        $conferenceData->subscribers  = ",$userID,";
        $conferenceData->participants = $type == 'default' ? ",$userID," : '' ;
        $conferenceData->invitee      = empty($invitee) ? '' : ",$invitee,";
        $conferenceData->topic        = $topic;
        $conferenceData->type         = $type;
        $conferenceData->password     = $password;
        if(!empty($startTime)) $conferenceData->startTime    = $startTime;
        if(!empty($endTime)) $conferenceData->endTime      = $endTime;
        $conferenceData->number       = $number;
        $conferenceData->reminderTime = $reminderTime;
        $conferenceData->note         = $note;

        $this->dao->update(TABLE_IM_CONFERENCE)->data($conferenceData)->where('id')->eq($id)->exec();
        $conference = $conferenceData;

        $this->saveAction($conference->rid, 'create', $userID);

        if(dao::isError()) return false;
        return $conference;
    }

    /**
     * set the scheduled conference status to open, and return the conference.
     * @param $conference
     * @return object
     */
    public function startSchedule($conference)
    {
        $this->dao->update(TABLE_IM_CONFERENCE)
                  ->set('status')->eq('open')
                  ->where('number')->eq($conference->number)
                  ->exec();
        if(dao::isError()) return false;
        return $this->dao->select('*')->from(TABLE_IM_CONFERENCE)
                         ->where('number')->eq($conference->number)
                         ->fetch();
    }

    /**
     * Create or activate a conference.
     *
     * @deprecated
     * @param  string         $chatID
     * @param  string         $invitee
     * @param  int            $userID
     * @access public
     * @return object|boolean
     */
    public function create($chatID, $invitee, $userID = 0)
    {
        $this->loadModel('conference');

        $date = helper::now();
        $conference = $this->getByChatID($chatID);

        /* Set default values for detached conference. */
        $startTime = null;
        $endTime   = null;
        $realName  = $this->loadModel('im')->userGetByID($userID)->realname;
        $topic     = sprintf($this->lang->im->conference->defaultTopic, $realName);
        $type      = 'default';
        $password  = '';

        if(!empty($conference)) $room = $this->conference->getRoom($conference->rid);
        if(!empty($conference) && !empty($room))
        {
            if($conference->status == 'open')
            {
                $participantAddition = $this->addParticipant($chatID, $userID);
                if(!$participantAddition) return $conference;

                $conference->participants = $participantAddition;
                $this->saveAction($conference->rid, 'join', $userID);
                return $conference;
            }
            $conference->status       = 'open';
            $conference->participants = ",$userID,";
            $conference->openedDate   = helper::now();
            $conference->openedBy     = $userID;
            $conference->invitee      = empty($invitee) ? '' : ",$invitee,";
            $conference->topic        = $topic;
            $conference->type         = $type;
            $conference->password     = $password;
            $conference->startTime    = $startTime;
            $conference->endTime      = $endTime;
            $this->dao->update(TABLE_IM_CONFERENCE)
                ->set('status')->eq($conference->status)
                ->set('participants')->eq($conference->participants)
                ->set('openedDate')->eq($conference->openedDate)
                ->set('openedBy')->eq($conference->openedBy)
                ->set('invitee')->eq($conference->invitee)
                ->set('topic')->eq($conference->topic)
                ->set('type')->eq($conference->type)
                ->set('password')->eq($conference->password)
                ->set('startTime')->eq($conference->startTime)
                ->set('endTime')->eq($conference->endTime)
                ->where('id')->eq($conference->id)
                ->exec();
        }
        else
        {
            $rid = str_replace('&', '-', $chatID);
            $conferenceConfig = $this->conference->getConfiguration();
            if(!isset($conferenceConfig->backendtype) || $conferenceConfig->backendtype == 'owt')
            {
                $roomInfo = $this->conference->createRoom($chatID);
                if(empty($roomInfo)) return false;
                $rid = $roomInfo->id;
            }

            $conferenceData = new stdClass();
            $conferenceData->cgid         = $chatID;
            $conferenceData->rid          = $rid;
            $conferenceData->status       = 'open';
            $conferenceData->participants = ",$userID,";
            $conferenceData->openedBy     = (int)$userID;
            $conferenceData->openedDate   = $date;
            $conferenceData->invitee      = empty($invitee) ? '' : ",$invitee,";
            $conferenceData->topic        = $topic;
            $conferenceData->type         = $type;
            $conferenceData->password     = $password;
            if(!empty($startTime)) $conferenceData->startTime    = $startTime;
            if(!empty($endTime)) $conferenceData->endTime      = $endTime;

            if(empty($conference)) $this->dao->insert(TABLE_IM_CONFERENCE)->data($conferenceData)->exec();
            else                   $this->dao->update(TABLE_IM_CONFERENCE)->data($conferenceData)->where('cgid')->eq($chatID)->exec();

            $conference = $conferenceData;
        }

        $this->saveAction($conference->rid, 'create', $userID);

        if(dao::isError()) return false;
        return $conference;
    }

    /**
     * Close a conference.
     *
     * @param  string  $conferenceId
     * @param  int     $userID
     * @access public
     * @return boolean
     */
    public function close($conferenceId, $userID)
    {
        $conference = $this->getById($conferenceId);

        if($conference->status == 'closed') return true;

        $this->dao->update(TABLE_IM_CONFERENCE)
            ->set('status')->eq('closed')
            ->set('participants')->eq('')
            ->where('id')->eq($conference->id)
            ->exec();

        $this->saveAction($conference->rid, 'close', $userID);
        $this->loadModel('conference')->deleteRoom($conference->rid);

        return !dao::isError();
    }

    /**
     * hide a conference.
     *
     * @param  string $conferenceNumber
     * @param  int    $userID
     * @access public
     * @return bool
     */
    public function hide($conferenceNumber, $userID)
    {
        $conferenceId = $this->extractIdFromNumber($conferenceNumber);
        $data = $this->dao->select('*')->from(TABLE_IM_CONFERENCEUSER)->where('conference')->eq($conferenceId)->andWhere('user')->eq($userID)->fetch();
        if(empty($data))
        {
            $this->dao->insert(TABLE_IM_CONFERENCEUSER)
                ->set('conference')->eq($conferenceId)
                ->set('user')->eq($userID)
                ->set('hide')->eq(1)
                ->exec();
        }
        else
        {
            $this->dao->update(TABLE_IM_CONFERENCEUSER)
                ->set('hide')->eq(1)
                ->where('conference')->eq($conferenceId)
                ->andWhere('user')->eq($userID)
                ->exec();
        }

        return !dao::isError();
    }

    /**
     * update a conference.
     *
     * @param  object $conference
     * @return bool
     */
    public function update($conference)
    {
        $this->dao->update(TABLE_IM_CONFERENCE)
            ->data($conference)
            ->where('number')->eq($conference->number)
            ->exec();

        return !dao::isError();
    }

    /**
     * after sent notify, update the sentNotify field.
     * @param object $conference
     * @param int $times
     * @return void
     */
    public function updateSentNotifyTimes($conference, $times) {
        $this->dao->update(TABLE_IM_CONFERENCE)
                  ->set('sentNotify')->eq($times)
                  ->where('number')->eq($conference->number)
                  ->exec();
    }

    /**
     * Add invitee into a conference.
     *
     * @param  string        $chatIdOrNumber
     * @param  array         $newInvitee
     * @access public
     * @return string|boolean
     */
    public function addInvitee($chatIdOrNumber, $newInvitee)
    {
        if(is_numeric($chatIdOrNumber))
        {
            $conference = $this->getByNumber($chatIdOrNumber);
        }
        else
        {
            $conference = $this->getByChatID($chatIdOrNumber);
        }

        if($conference->status == 'closed') return false;

        $invitee = explode(',', $conference->invitee);
        $invitee = array_filter($invitee);
        $invitee = array_merge($invitee, $newInvitee);
        $invitee = array_unique($invitee);
        $invitee = implode(',', $invitee);
        $invitee = empty($invitee) ? '' : ",$invitee,";

        $this->dao->update(TABLE_IM_CONFERENCE)
            ->set('invitee')->eq($invitee)
            ->where('id')->eq($conference->id)
            ->exec();

        if(dao::isError()) return false;
        return $invitee;
    }

    /**
     * Add participant into a conference.
     *
     * @param  string        $chatIdOrNumber
     * @param  int           $userID
     * @access public
     * @return string|boolean
     */
    public function addParticipant($chatIdOrNumber, $userID)
    {
        if(is_numeric($chatIdOrNumber))
        {
            $conference = $this->getByNumber($chatIdOrNumber);
        }
        else
        {
            $conference = $this->getByChatID($chatIdOrNumber);
        }

        if($conference->status == 'closed') return false;

        $participants = explode(',', $conference->participants);
        $participants = array_filter($participants);

        if(!$this->isUnlimitedParticipants() && count($participants) > 2) return false;

        $participants[] = $userID;
        $participants = array_unique($participants);
        $participants = implode(',', $participants);
        $participants = empty($participants) ? '' : ",$participants,";

        $this->dao->update(TABLE_IM_CONFERENCE)
            ->set('participants')->eq($participants)
            ->where('id')->eq($conference->id)
            ->exec();

        if(dao::isError()) return false;
        return $participants;
    }

    /**
     * Remove participant from a conference.
     *
     * @param  string        $chatIdOrNumber
     * @param  int           $userID
     * @access public
     * @return array|boolean
     */
    public function removeParticipant($chatIdOrNumber, $userID)
    {
        if(is_numeric($chatIdOrNumber))
        {
            $conference = $this->getByNumber($chatIdOrNumber);
        }
        else
        {
            $conference = $this->getByChatID($chatIdOrNumber);
        }

        if($conference->participants == '') return false;

        $participants = explode(',', $conference->participants);
        $participants = array_filter($participants);
        $participants = array_diff($participants, array($userID));
        $participants = implode(',', $participants);
        $participants = empty($participants) ? '' : ",$participants,";

        $this->dao->update(TABLE_IM_CONFERENCE)
            ->set('participants')->eq($participants)
            ->where('id')->eq($conference->id)
            ->exec();

        if(dao::isError()) return false;
        return $participants;
    }

    /**
     * Get conference actions since last close.
     *
     * @param  object $conference
     * @param  string $type
     * @access public
     * @return boolean
     */
    public function getActions($conference, $type = '')
    {
        $actions = $this->dao->select('*')->from(TABLE_IM_CONFERENCEACTION)
            ->where('rid')->eq($conference->rid)
            ->andWhere('date')->gt($conference->openedDate)
            ->beginIF($type)->andWhere('type')->eq($type)->fi()
            ->orderBy('id')
            ->fetchAll();

        if(!empty($actions))
        {
            foreach($actions as $action)
            {
                $action->user = (int)$action->user;
                $action->date = strtotime($action->date);
                $action->room = $action->rid;
                unset($action->rid);
            }
        }

        if(dao::isError()) return false;
        return $actions;
    }

    /**
     * Save a conference action.
     *
     * @param  string         $roomID
     * @param  string         $type
     * @param  int            $userID
     * @param  string         $device
     * @access public
     * @return object|boolean
     */
    public function saveAction($roomID, $type, $userID, $device = 'default', $data = '')
    {
        $action = new stdClass();
        $action->rid    = $roomID;
        $action->type   = $type;
        $action->user   = $userID;
        $action->device = $device;
        $action->date   = helper::now();
        $action->data   = $data;
        $this->dao->insert(TABLE_IM_CONFERENCEACTION)->data($action)->exec();

        if ($type == 'join')
        {
            $conference = $this->dao->select('subscribers')->from(TABLE_IM_CONFERENCE)
                          ->where('rid')->eq($roomID)
                          ->fetch();
            if (!empty($conference))
            {
                $subscribers = explode(',', $conference->subscribers);
                $subscribers = array_filter($subscribers);
                $subscribers = array_unique(array_merge($subscribers, array($userID)));
                $subscribers = ',' . implode(',', $subscribers) . ',';
                $this->dao->update(TABLE_IM_CONFERENCE)
                    ->set('subscribers')->eq($subscribers)
                    ->where('rid')->eq($roomID)
                    ->exec();
            }
        }

        if(dao::isError()) return false;

        $action->room = $action->rid;
        $action->date = strtotime($action->date);
        return $action;
    }

    /**
     * Check if user is occupied.
     *
     * @param  int     $userID
     * @param  string  $fromChat  from which chat check user status
     * @access public
     * @return boolean
     */
    public function isUserOccupied($userID, $fromChat = '')
    {
        /* TODO: filter with sql like in 7.2.stable */
        $conferences = $this->dao->select('*')->from(TABLE_IM_CONFERENCE)
            ->where('status')->eq('open')
            ->fetchAll();
        foreach($conferences as $conference)
        {
            $participants = explode(',', $conference->participants);
            $participants = array_filter($participants);
            if((!$fromChat || $conference->cgid != $fromChat) && in_array($userID, $participants)) return true;
        }
        return false;
    }

    /**
     * Remove the user from all related conferences and close if necessary.
     *
     * @param  int  $userID
     * @access public
     * @return void
     */
    public function removeUserFromConferences($userID)
    {
        $conferences = $this->getByUserID($userID);

        foreach($conferences as $conference)
        {
            $participants = explode(',', $conference->participants);
            $participants = array_filter($participants);
            if(in_array($userID, $participants))
            {
                if(count($participants) > 1 || $conference->type != 'default')
                {
                    $this->removeParticipant($conference->cgid, $userID);
                    $this->saveAction($conference->rid, 'leave', $userID);
                }
                else
                {
                    $this->close($conference->id, $userID);
                }
            }
        }
    }

    /**
     * Try to remove user from a conference of a chat.
     *
     * @deprecated
     * @param  string $chatID
     * @param  int    $userID
     * @access public
     * @return object|boolean
     */
    public function removeUserFromChat($chatID, $userID)
    {
        /* Ignore conference number. */
        if(is_numeric($chatID)) return false;

        $conference = $this->getByChatID($chatID);
        if(empty($conference) || $conference->status != 'open') return false;

        $participants = explode(',', $conference->participants);
        $participants = array_filter($participants);
        $participants = array_diff($participants, array($userID));

        if(count($participants) == 0)
        {
            $this->close($conference->id, $userID);
            return false;
        }

        $participants = implode(',', $participants);
        $conference->participants = $participants;

        return $conference;
    }

    /**
     * Clean conferences participants and close conference if needed or return the conference.
     *
     * @deprecated
     * @param  string         $chatID
     * @param  int            $userID
     * @access public
     * @return object|boolean
     */
    public function cleanChatConference($chatID, $userID)
    {
        $conference = $this->getByChatID($chatID);
        if(empty($conference)) return false;

        $participants = explode(',', $conference->participants);
        $participants = array_filter($participants);

        $onlineUsers = array_keys($this->loadModel('im')->user->getList('online'));
        $participants = array_intersect($participants, $onlineUsers);

        if(current($participants) == $userID)
        {
            $this->close($conference->id, $userID);
            return false;
        }

        $participants = implode(',', $participants);
        $conference->participants = $participants;

        return $conference;
    }

    /**
    * Clean detached conferences participants and close conference if needed or return the conference.
    * if conference participants is empty or offline, close it.
    *
    * @param  string         $chatID
    * @param  int            $userID
    * @access public
    * @return object|boolean
    */
    public function cleanDetachedChatConference($chatID, $userID)
    {
        $conferences = $this->dao->select('*')->from(TABLE_IM_CONFERENCE)
                                 ->where('number')->ne('')
                                 ->andWhere('cgid')->eq($chatID)
                                 ->andWhere('status')->eq('open')
                                 ->andWhere('type')->eq('default')
                                 ->orderBy('id_asc')->fetchAll();

        if(empty($conferences)) return false;

        $onlineUsers = array_keys($this->loadModel('im')->user->getList('online'));

        $result = array();
        foreach($conferences as $conference)
        {
            $participants = explode(',', $conference->participants);
            $participants = array_filter($participants);
            $participants = array_intersect($participants, $onlineUsers);

            if(empty($participants) || current($participants) == $userID)
            {
                $this->close($conference->id, $userID);
            }
            else
            {
                $result[] = $conference;
            }
        }

        return $result;
    }

    public function cleanDetachedConference($userID)
    {
        $conferences = $this->dao->select('*')->from(TABLE_IM_CONFERENCE)
                                 ->where('number')->ne('')
                                 ->andWhere('cgid')->eq('')
                                 ->orderBy('id_asc')->fetchAll();

        if(empty($conferences)) return false;

        $onlineUsers = array_keys($this->loadModel('im')->user->getList('online'));

        $result = array();
        foreach($conferences as $conference)
        {
            $participants = explode(',', $conference->participants);
            $participants = array_filter($participants);
            $participants = array_intersect($participants, $onlineUsers);

            if(empty($participants) || current($participants) == $userID)
            {
                $this->close($conference->id, $userID);
            }
            else
            {
                $result[] = $conference;
            }
        }

        return $result;
    }

    /**
     * Reset status of conferences.
     *
     * @access public
     * @return void
     */
    public function resetStatus()
    {
        $this->dao->update(TABLE_IM_CONFERENCE)
            ->set('status')->eq('closed')
            ->set('participants')->eq('')
            ->exec();
    }

    /**
     * Get open conferences by given chat list.
     * in the meantime, clean the chat conference. close it if necessary.
     *
     * @param  array  $chatList
     * @param  int    $userID
     * @param  string $version
     * @access public
     * @return array
     */
    public function getAndCleanOpenConferences($chatList, $userID, $ignoreActions = false, $version = '')
    {
        $conferenceConfig = $this->loadModel('conference')->getConfiguration();
        if(isset($conferenceConfig->enabled) && $conferenceConfig->enabled == '0') return array();

        $openConferenceList = array();
        foreach($chatList as $chat)
        {
            /* TODO: Move clean function to xxd polling */
            $chatConference = $this->cleanChatConference($chat->gid, $userID);
            if(!empty($chatConference) && $chatConference->status == 'open')
            {
                $actions = $this->getActions($chatConference);
                $chatConference->actions = ($ignoreActions || empty($actions)) ? array() : $actions;

                $chatConference->room       = $chatConference->rid;
                $chatConference->openedBy   = (int)$chatConference->openedBy;
                $chatConference->openedDate = strtotime($chatConference->openedDate);
                unset($chatConference->rid);

                $openConferenceList[] = $chatConference;
            }

            if(!empty($version) && version_compare($version, '7.2.beta', '>=') && isset($conferenceConfig->detachedConference) && $conferenceConfig->detachedConference)
            {
                /* TODO: Move clean function to xxd polling */
                $detachedConferences = $this->cleanDetachedChatConference($chat->gid, $userID);
                if(!empty($detachedConferences))
                {
                    foreach($detachedConferences as $conference)
                    {
                        $actions = $this->getActions($conference);
                        $conference->actions    = ($ignoreActions || empty($actions)) ? array() : $actions;
                        $conference->room       = $conference->rid;
                        $conference->openedBy   = (int)$conference->openedBy;
                        $conference->openedDate = strtotime($conference->openedDate);
                        unset($conference->rid);

                        $openConferenceList[] = $conference;
                    }
                }
            }
        }

        /* Clean none chat detached conferences */
        $detachedConferences = $this->cleanDetachedChatConference('', $userID);
        if(!empty($detachedConferences))
        {
            foreach($detachedConferences as $conference)
            {
                $actions = $this->getActions($conference);
                $conference->actions    = ($ignoreActions || empty($actions)) ? array() : $actions;
                $conference->room       = $conference->rid;
                $conference->openedBy   = (int)$conference->openedBy;
                $conference->openedDate = strtotime($conference->openedDate);
                unset($conference->rid);

                $openConferenceList[] = $conference;
            }
        }

        $thisClass = $this;
        return array_map(function($conference) use ($thisClass) { return $thisClass->format($conference); }, $openConferenceList);
    }

    /**
     * Get all open conferences.
     *
     * @access public
     * @return array
     */
    public function getOpenConferences()
    {
        return $this->dao->select('*')->from(TABLE_IM_CONFERENCE)
            ->where('status')->eq('open')
            ->fetchAll();
    }

    /**
     * Check if there is participant limit in license.
     *
     * @access public
     * @return string|bool
     */
    public function isUnlimitedParticipants()
    {
        return extCommonModel::getLicensePropertyValue('unlimitedParticipants');
    }
}
