#!/bin/bash
set -e
# shellcheck disable=SC1091
. /opt/zbox/bin/lib/liblog.sh

# zentao version
zentaoVer=(
  "zentao"
  "zentaoep"
  "zentaomax"
  "zentaoipd"
  )
export zentaoVer

# Get ZenTao Type
function get_zentao_type(){
    local ver=${1:?version is required!}
    echo "zentao"
#
#    case $ver in
#    biz*)
#        echo "zentaoep"
#    ;;
#    max*)
#        echo "zentaomax"
#    ;;
#    ipd*)
#        echo "zentaoipd"
#        ;;
#    *)
#        echo "zentao"
#    ;;
#    esac
}

# get zentao version
if [ -f /opt/zbox/VERSION ];then
  ZT_VER=$(cat < /opt/zbox/VERSION | head -n 1)
fi
export ZT_VER=${ZT_VER:-"zentao"}
ZT_TYPE=$(get_zentao_type "${ZT_VER:-zentao}")
export ZT_TYPE=${ZT_TYPE:-"zentao"}
export RR_PROC="/opt/zbox/bin/rr serve -w /opt/zbox/app/$ZT_TYPE/roadrunner"
export REDIS_PROC="/opt/zbox/run/redis/redis-server /opt/zbox/etc/redis/redis.conf"

###############################################################
# Function: check_basePath
# Description: Check the root directory
# Arguments:
# $1 - Base directory path
# Returns: None
###############################################################
check_basePath(){
  base=$1
  if [ "$base" != "/opt/zbox" ];then
    error "Run $0 in the /opt/zbox directory"
    exit 1
  fi
}

###############################################################
# Function: check_permission
# Description: Change directory permissions
# Arguments:
# $1 - User
# $2 - Group
# Returns: None
###############################################################
function check_permission(){
    local user=${1?user is required}
    local group=${2?group is required}

    # 更改禅道代码目录权限
    for dir in "${zentaoVer[@]}"
    do
      if [ -d /opt/zbox/app/"$dir" ];then
        chmod -R 777 /opt/zbox/app/"$dir"/tmp
        chmod -R 777 /opt/zbox/app/"$dir"/www/data
        chmod 777 /opt/zbox/app/"$dir"/www/
        chmod 777 /opt/zbox/app/"$dir"/config/
        chmod -R a+rx /opt/zbox/app/"$dir"/bin/*
      fi
    done

    chmod -R 777 /opt/zbox/tmp
    chown -R "$user":"$group" /opt/zbox/data/mysql /opt/zbox/logs
    chown -R "$user":"$group" /opt/zbox/app/*
}

###############################################################
# Function: control_apache
# Description: Control Apache service
# Arguments:
# $1 - Action (start, stop, restart)
# Returns: None
###############################################################
function control_apache(){
  local action=$1
  local port=$2
  # 转换为小写
  action=${action,,}

  status=$(get_service_status '/opt/zbox/run/apache/httpd')

  case $action in
  "start")
    if [ "$status" == "running" ];then
      warn "Apache is running."
    elif [ "$(get_service_port "$port")" == "1" ];then
      error "Apache $port port is already in use."
      exit 1
    else
      if [ "${APACHE_SERVICE,,}" == "enabled" ];then
        info "Starting apache service..."
        /opt/zbox/run/apache/apachectl start || error "Apache start failure."
      else
        warn "Apache serivce is disabled."
      fi
    fi
  ;;
  "stop")
    if [ "$status" == "running" ];then
      info "Stopping the apache service..."
      /opt/zbox/run/apache/apachectl stop

      # wait for stop apache
      wait_apache_stop

    else
      if [ "${APACHE_SERVICE,,}" != "enabled" ];then
        warn "Apache service is disabled."
      else
        warn "Apache service is not running."
      fi
    fi
  ;;
  "check")
    if [ "${APACHE_SERVICE,,}" != "enabled" ];then
      control_apache "stop"
    else
      control_apache "start" "$port"
    fi
    ;;
  "restart")
    control_apache "stop" && control_apache "start" "$port"
  ;;
  esac
}

###############################################################
# Function: check_and_repair_db
# Description: Check and repair mysql database
# Arguments: None
# Returns: None
###############################################################
function check_and_repair_db(){

  if [ ! -f /opt/zbox/run/mysql/mariadb-check ] || [ ! -f /opt/zbox/run/mysql/mariadb-upgrade ];then
    warn "MySQL check commands not exist skip database check."
    return 0
  fi

  info "Checking MySQL database ..."
  IsOK=$(/opt/zbox/run/mysql/mariadb-check -c -uroot -p"$MYSQL_PASS" -S "$MYSQL_SOCK" -A | grep -v OK || true)

  if [ "$IsOK" != "" ];then
    info "Optimizing MySQL database ..."
	  /opt/zbox/run/mysql/mariadb-upgrade -uroot -p"$MYSQL_PASS" -S "$MYSQL_SOCK" --force  > /dev/null 2>&1
  fi
}

###############################################################
# Function: control_mysql
# Description: Control MySQL service
# Arguments:
# $1 - Action (start, stop, restart)
# Returns: None
###############################################################
function control_mysql(){
  local action=$1
  # 转换为小写
  action=${action,,}

  status=$(get_service_status '/opt/zbox/run/mysql/mariadbd')

  case $action in
  "start")
    if [ "$status" == "running" ];then
      warn "MySQL is running."
    elif [ "$(get_service_port "$mport")" == "1" ];then
      error "MySQL $mport port is already in use."
      exit 1
    else
      if [ "${MYSQL_SERVICE,,}" == "enabled" ];then
        # set mysql innodb buffuer pool size
        set_mysql_buffer_pool

        info "Starting mysql service..."
        # start mysql
        bash /opt/zbox/run/mysql/mysql.server start > /dev/null 2>&1

        # check mysql database
        if [ "$CHECK_MYSQL" == "true" ];then
          check_and_repair_db
        fi
      else
        warn "MySQL service is disabled."
      fi
    fi
  ;;
  "stop")
    if [ "$status" == "running" ];then
      info "Stopping the mysql service..."
      /opt/zbox/run/mysql/mysqladmin shutdown -uroot -p"$MYSQL_PASS" -S /opt/zbox/tmp/mysql/mysql.sock > /dev/null 2>&1 || error "Failed to shutdown MySQL."
      wait_mysql_stop
    else
      if [ "${MYSQL_SERVICE,,}" != "enabled" ];then
        warn "MySQL service is disabled."
      else
        warn "MySQL service is not running."
      fi
    fi
  ;;
  "check")
    if [ "${MYSQL_SERVICE,,}" != "enabled" ];then
      control_mysql "stop"
    else
      control_mysql "start"
    fi
    ;;
  "restart")
    control_mysql "stop" && control_mysql "start"
  ;;
  esac
}

###############################################################
# Function: control_xxd
# Description: Control XXD service
# Arguments:
# $1 - Action (start, stop, restart)
# Returns: None
###############################################################
function control_xxd(){
  local action=$1
  # 转换为小写
  action=${action,,}

  status=$(get_service_status '/opt/zbox/run/xxd/xxd')

  case $action in
  "start")
    if [ "$status" == "running" ];then
      warn "XXD is running."
    else
      if [ "${XXD_SERVICE,,}" == "enabled" ];then
        info "Starting xxd service..."
        # 首次启动，配置xxd
        init_xxd

        # 启动xxd
        nohup /opt/zbox/run/xxd/xxd > /opt/zbox/logs/xxd.log 2>&1 &
        disown
      else
        warn "XXD service is disabled."
      fi
    fi
  ;;
  "stop")
    if [ "$status" == "running" ];then
      info "Stopping the xxd service..."
      kill -9 $(pgrep -f '/opt/zbox/run/xxd/xxd')
    else
      if [ "${XXD_SERVICE,,}" != "enabled" ];then
        warn "XXD service is disabled."
      else
        warn "XXD service is not running."
      fi
    fi
  ;;
  "check")
    if [ "${XXD_SERVICE,,}" != "enabled" ];then
      control_xxd "stop"
    else
      control_xxd "start"
    fi
    ;;
  "restart")
    control_xxd "stop" && control_xxd "start"
  ;;
  esac
}

###############################################################
# Function: wait_mysql_start
# Description: Wait for MySQL to start
# Arguments: None
# Returns: None
# Note: 30 seconds timeout
###############################################################
function wait_mysql_start(){

  for ((i = 1; i <= 30; i += 1)); do
    if /opt/zbox/run/mysql/mysqladmin ping -uroot -p"$MYSQL_PASS" -h"$MYSQL_HOST" -P"$mport" > /dev/null 2>&1;
    then
      info "MySQL start successfully."
      break
    else
      info "Waiting for start MySQL $i seconds ..."
    fi
    sleep 1
  done
}

###############################################################
# Function: wait_mysql_stop
# Description: Wait for MySQL to stop
# Arguments: None
# Returns: None
# Note: 30 seconds timeout
###############################################################
function wait_mysql_stop(){

  for ((i = 1; i <= 30; i += 1)); do
    if /opt/zbox/run/mysql/mysqladmin ping -uroot -p"$MYSQL_PASS" -h"$MYSQL_HOST" -P"$mport" > /dev/null 2>&1;
    then
      info "Waiting for shutdown MySQL $i seconds ..."
      # 捡漏，pkill清理zbox中的mysql进程
      pkill -f '/opt/zbox/run/mysql'
    else
      info "MySQL stopped successfully."
      break
    fi
    sleep 1
  done
}

###############################################################
# Function: wait_apache_stop
# Description: Wait for Apache to stop
# Arguments: None
# Returns: None
# Note: 30 seconds timeout
###############################################################
function wait_apache_stop(){

  for ((i = 1; i <= 30; i += 1)); do
    if pgrep -f '/opt/zbox/run/apache/httpd' > /dev/null;
    then
      info "Waiting for stop apache $i seconds ..."

      # 捡漏，pkill清理zbox中的mysql进程
      /opt/zbox/run/apache/apachectl stop > /dev/null 2>&1
    else
      info "Apache stopped successfully."
      break
    fi
    sleep 1
  done

}

###############################################################
# Function: check_user_group
# Description: Check if user and group exist, create them if not
# Arguments:
# $1 - User
# $2 - Group
# Returns: None
###############################################################
function check_user_group(){
    local user=${1?user is required}
    local group=${2?group is required}
  # 不存在账号，创建账号
  if ! getent passwd "$user" > /dev/null 2>&1;then
    useradd "$user"
  fi

  # 不存在组，创建组
  if ! getent group "$group" > /dev/null 2>&1;then
    groupadd "$group"
  fi
}

###############################################################
# Function: service_status
# Description: Print service status information
# Arguments: None
# Returns: None
###############################################################
function service_status(){
    apacheVer=$(/opt/zbox/bin/apachectl -v| grep -o -E '[0-9]+\.[0-9]+\.[0-9]+')
    apachePort=$(grep Listen /opt/zbox/etc/apache/httpd.conf | awk '{print $2}')
    xxdVer=$(/opt/zbox/run/xxd/xxd -V|awk '{gsub("v","",$2); print $2}')

    mysqlVer=$(/opt/zbox/run/mysql/mariadbd -V | awk '{print $3}'|sed -r 's/\-.*//g')
    mysqlPort=$(grep  '\[mysqld\]' -A10  /opt/zbox/etc/mysql/my.cnf  | awk '{if ($1 == "port") print $3}')

    phpVer=$(/opt/zbox/run/php/php -v| grep ^PHP | grep -E -o '[0-9]+\.[0-9]+\.[0-9]+')
    apacheMPM=$(/opt/zbox/run/apache/apachectl -V | grep "Server MPM"| awk '{print $3}')
    xxdPort=$(grep -E '(common|chat)Port' /opt/zbox/run/xxd/config/xxd.conf | awk '{print $3}' | xargs)

    rrVer=$(/opt/zbox/bin/rr -v | awk '{print $3}')

    # redis
    [ -f /opt/zbox/etc/redis/redis.conf ] && redisPort=$(grep port /opt/zbox/etc/redis/redis.conf | grep -E -o '[0-9]+')
    [ -f /opt/zbox/bin/redis-server ] && redisVer=$(/opt/zbox/bin/redis-server  -v | awk '{print $3}'| grep -E -o '[0-9]+\.[0-9]+\.[0-9]+')

    print_line
    printf "%-20s%-15s%-10s%-10s%-15s\n" "Service" "Port" "Status" "Version" "Comment"
    print_line
    if [ "${APACHE_SERVICE,,}" == "enabled" ];then
      print_service_status "Apache" "$apachePort" "$(get_service_status '/opt/zbox/run/apache/httpd')" "$apacheVer" ""
    else
      print_service_status "Apache" "$apachePort" "disabled" "$apacheVer" ""
    fi
    print_service_status "PHP" "-" "$(get_service_status '/opt/zbox/run/apache/httpd')" "$phpVer" "$apacheMPM"

    if [ "${MYSQL_SERVICE,,}" == "enabled" ];then
      print_service_status "MySQL" "$mysqlPort" "$(get_service_status '/opt/zbox/run/mysql/mariadbd')" "$mysqlVer" "MariaDB"
    else
      print_service_status "MySQL" "$mysqlPort" "disabled" "$mysqlVer" "MariaDB"
    fi

    if [ "${XXD_SERVICE,,}" == "enabled" ];then
      print_service_status "XXD" "$xxdPort" "$(get_service_status '/opt/zbox/run/xxd/xxd')" "${xxdVer:-unknown}" ""
    else
      print_service_status "XXD" "$xxdPort" "disabled" "${xxdVer:-unknown}" ""
    fi

    if [ "${ROADRUNNER_SERVICE,,}" == "enabled" ];then
      print_service_status "Roadrunner" "-" "$(get_service_status "$RR_PROC")" "${rrVer:-unknown}" "@$ZT_TYPE"
    else
      print_service_status "Roadrunner" "-" "disabled" "${rrVer:-unknown}" "@$ZT_TYPE"
    fi

    if [ "${REDIS_SERVICE,,}" == "enabled" ];then
      print_service_status "Redis" "$redisPort" "$(get_service_status "/opt/zbox/run/redis/redis-server")" "${redisVer:-unknown}" "Cache"
    # else
    #   print_service_status "Redis" "$redisPort" "disabled" "${redisVer:-unknown}" "Cache"
    fi

    echo
    print_line
    printf "%-20s%-15s\n" "Application" "Version"
    print_line
    print_app_status
}

###############################################################
# Function: print_service_status
# Description: Print service status line
# Arguments:
# $1 - Service name
# $2 - Port
# $3 - Status
# $4 - Version
# $5 - Comment
# Returns: None
###############################################################
function print_service_status(){
  local service=${1?service is required }
  local port=${2?port is required }
  local status=${3?status is required }
  local ver=${4?version is required }
  local comment=$5
  if [ "$status" == "running" ];then
    printf "%-20s%-15s${GREEN}%-10s${RESET}%-10s%-15s\n" "$service" "$port" "$status" "$ver" "$comment"
  elif [ "$status" == "stopped" ];then
    printf "%-20s%-15s${RED}%-10s${RESET}%-10s%-15s\n" "$service" "$port" "$status" "$ver" "$comment"
  elif [ "$status" == "disabled" ];then
    printf "%-20s%-15s${GRAY}%-10s${RESET}%-10s%-15s\n" "$service" "$port" "$status" "$ver" "$comment"
  else
    printf "%-20s%-15s${GREEN}%-10s${RESET}%-10s%-15s\n" "$service" "$port" "$status" "$ver" "$comment"
  fi
}

###############################################################
# Function: print_app_status
# Description: Print application status
# Arguments: None
# Returns: None
###############################################################
function print_app_status(){
  for app in "${zentaoVer[@]}"
  do
    if [ -f /opt/zbox/app/"$app"/VERSION ];then
      ver=$(cat /opt/zbox/app/"$app"/VERSION)
      printf "%-20s%-15s\n" "$app" "$ver"
    fi
  done

  adminerVer=$(head -n10 /opt/zbox/app/ztadmin/index.php |grep @version | awk '{print $3}')
  printf "%-20s%-15s\n" "adminer" "$adminerVer"
}

###############################################################
# Function: get_service_status
# Description: Get service status (running or stopped)
# Arguments:
# $1 - Program info
# Returns: "running" if the service is running, "stopped" otherwise
###############################################################
function get_service_status(){
  local info=${1?program info is required}

  if pgrep -f "$info" > /dev/null ;then
    echo "running"
  else
    echo "stopped"
  fi
}

###############################################################
# This function checks if a service is running on a specific port.
# If a service is running on the specified port, it returns 1. Otherwise, it returns 0.
# Parameters:
#   $1: The port number to check.
# Usage:
#   get_service_port 8080
###############################################################
function get_service_port(){
  local port=${1?program port is required}
  isExist=$(ss -tln sport = :"$port" | sed 1d 2>/dev/null |wc -l)

  if [ "$isExist" == "0" ];then
    echo "0"
  else
    echo "1"
  fi
}

###############################################################
# Function: change_port
# Description: Change the port of a service
# Arguments:
# $1 - Service name (apache, mysql)
# $2 - New port number
# Returns: None
###############################################################
function change_port(){
  service=$1
  port=$2
  case $service in
  "apache")
      sed -i.bak."$(date +%Y%m%d%H%M)" -r \
          -e "s/(^Listen) .*/\1 $port/g" \
          -e "s/<(VirtualHost \*\:).*>/\1$port/g" \
          /opt/zbox/etc/apache/httpd.conf
      # change .env file
      sed -i.bak -r -e "s/(APACHE_PORT)=.*/\1=$port/g" /opt/zbox/.env

      # change xxd config
      sed -i.bak -r -e "s#(http://${APACHE_HOST})(:[0-9]+){0,1}#\1:${port}#g" /opt/zbox/run/xxd/config/xxd.conf
  ;;
  "mysql")
      sed -i.bak."$(date +%Y%m%d%H%M)" -r \
          "s/(^port.*)=.*/\1= $port/g" \
          /opt/zbox/etc/mysql/my.cnf
      # change .env file
      sed -i.bak -r -e "s/(MYSQL_PORT)=.*/\1=$port/g" /opt/zbox/.env

      # change zentao my.php
      change_zentao_config "$port"
  ;;
  esac
}

######################################################################
# Function: change_zentao_config
# Description: Change the MySQL port in the Zentao configuration file
# Arguments:
# $1 - New port number
# Returns: None
######################################################################
function change_zentao_config(){
  port=$1

  for dir in "${zentaoVer[@]}"
  do
    if [ -f /opt/zbox/app/"$dir"/config/my.php ];then
        sed -i.bak."$(date +%Y%m%d%H%M)" -r \
        "s/(.*config->db->port .*)=.*/\1 = \'$port\';/g" \
        /opt/zbox/app/"$dir"/config/my.php
    fi
  done
}
###############################################################
# Function: config_xxd
# Description: Config xxd
# Arguments:
# $1 - type (zentao, biz, max)
# $2 - dbname
# Returns: None
# Note: This function will be called when the zbox is started
#       and the database is ready.
#       It will get the xxd token from the database and update the
#       xxd.conf file.
#       The xxd.conf file is used by the xxd service.
#       The xxd service will be started after the xxd.conf file is updated.
###############################################################
function config_xxd(){
  local type=$1
  dbname=${2:-zentao}

  MYSQL_BIN="/opt/zbox/bin/mysql -uroot -p$MYSQL_PASS -P$mport -h$MYSQL_HOST $dbname"

  # get xxd token from database and update xxd.conf file
  # retry 3 times
  for ((i = 1;i<=3;i++))
  do
    info "Get xxd token from $type"
    /opt/zbox/bin/php /opt/zbox/bin/curl.php http://"$APACHE_HOST":"$aport"/"$type" > /dev/null 2>&1
    xxdToken=$($MYSQL_BIN -e "select value from zt_config where section='xuanxuan' and \`key\`='key';" | sed 1d)
    if [ "$xxdToken" != "" ];then
      # enable xxd
      $MYSQL_BIN -e "update zt_config set \`value\`='1' where section='xuanxuan' and \`key\`='turnon';" > /dev/null 2>&1
      break
    else
      warn "Failed to get xxd token from $type, retry in $i second"
      sleep 1
    fi
  done

  if [ "$xxdToken" == "" ];then
    error "Failed to get xxd token from $type"
    exit 1
  fi

  sed -i "s/__${type^^}_TOKEN__/$xxdToken/g" /opt/zbox/run/xxd/config/xxd.conf
}

# Function: control_rr
# Description: Control roadrunner service
function control_rr(){
  local action=$1
  # 转换为小写
  action=${action,,}

  # get roadrunner pid
  rr_pid=$(get_service_status "$RR_PROC")

  case $action in
  "start")
    if [ "$rr_pid" == "running" ];then
      warn "Roadrunner is running."
    else
      if [ "${ROADRUNNER_SERVICE,,}" == "enabled" ];then
        info "Starting roadrunner service..."

        # 启动roadrunner
        nohup /opt/zbox/bin/rr serve -w /opt/zbox/app/"$ZT_TYPE"/roadrunner -p  > /opt/zbox/logs/rr.log 2>&1 &
        disown
      else
        warn "Roadrunner service is disabled."
      fi
    fi
  ;;
  "stop")
    if [ "$rr_pid" == "running" ];then
      info "Stopping the roadrunner service..."
      /opt/zbox/bin/rr stop -w /opt/zbox/app/"$ZT_TYPE"/roadrunner >> /opt/zbox/logs/rr.log 2>&1
    else
      if [ "${ROADRUNNER_SERVICE,,}" != "enabled" ];then
        warn "Roadrunner service is disabled."
      else
        warn "Roadrunner service is not running."
      fi
    fi
  ;;
  "check")
    if [ "${ROADRUNNER_SERVICE,,}" != "enabled" ];then
      control_rr "stop"
    else
      control_rr "start"
    fi
    ;;
  "restart")
    control_rr "stop"
    sleep 3
    control_rr "start"
  ;;
  esac
}

# Function: control_redis
# Description: Control redis service
function control_redis(){
  local action=$1
  # 转换为小写
  action=${action,,}

  # get roadrunner pid
  redis_pid=$(get_service_status "/opt/zbox/run/redis/redis-server")

  case $action in
  "start")
    if [ "$redis_pid" == "running" ];then
      warn "Redis is running."
    else
      if [ "${REDIS_SERVICE,,}" == "enabled" ];then
        info "Starting redis service..."

        # Start redis
        $REDIS_PROC
      else
        warn "Redis service is disabled."
      fi
    fi
  ;;
  "stop")
    if [ "$redis_pid" == "running" ];then
      info "Stopping the redis service..."
      /opt/zbox/bin/redis-cli SHUTDOWN
    else
      if [ "${REDIS_SERVICE,,}" != "enabled" ];then
        warn "Redis service is disabled."
      else
        warn "Redis service is not running."
      fi
    fi
  ;;
  "check")
    if [ "${REDIS_SERVICE,,}" != "enabled" ];then
      control_redis "stop"
    else
      control_redis "start"
    fi
    ;;
  "restart")
    control_redis "stop"
    sleep 3
    control_redis "start"
  ;;
  esac
}

###############################################################
# Function: init_xxd
# Description: Init xxd
# Arguments: None
# Returns: None
# Note: This function will be called when the zbox is started
# for the first time.
# /opt/zbox/run/xxd/config/xxd.conf and restart the zbox.
# The xxd.conf file is generated based on the zentao configuration file.
###############################################################
function init_xxd(){

  # 首次启动xxd
  isInit=$( sed  -n '/TOKEN__/=' /opt/zbox/run/xxd/config/xxd.conf 2>/dev/null )
  if [ "$isInit" != "" ]; then
    # 等待mysql就绪
    wait_mysql_start
    for d in /opt/zbox/app/zentao*
    do
      ztver=$(basename "$d")
      case $ztver in
      "zentao")
        config_xxd "zentao" "zentao"
        ;;
      "zentaoep")
        config_xxd "biz" "zentaoep"
        ;;
      "zentaomax")
        config_xxd "max" "zentaomax"
        ;;
      "zentaoipd")
        config_xxd "ipd" "zentaoipd"
        ;;
      esac
    done

    # 清理未替换的TOKEN
    sed -i '/TOKEN__/d' /opt/zbox/run/xxd/config/xxd.conf
  fi

}

###################################
# According to the total memory
# set MySQL Innodb buffer pool size
###################################
function set_mysql_buffer_pool(){

  # get inited identifiers
  inited=$(awk '/innodb_buffer_pool_size_inited/' /opt/zbox/etc/mysql/my.cnf 2>/dev/null)

  if [[ $inited == "" ]]; then

    # get system memory
    total_memory_by_KB=$(grep MemTotal /proc/meminfo | awk '{print $2}') 2>/dev/null

    # check total memory and set default memory size 4G
    total_memory_by_KB=${total_memory_by_KB:-409600}

    # 50% of memory(MB)
    pool_size=$((total_memory_by_KB/1024/2))

    if [[ $total_memory_by_KB -lt 1024000 ]]; then
      info "System memory size: $((total_memory_by_KB/1024))MB"
      info "Setting innodb buffer pool size : ${pool_size}MB"
    else
      info "System memory size: $((total_memory_by_KB/1024/1024))GB"
      info "Setting innodb buffer pool size : $((pool_size/1024))GB"
    fi 

    sed -i -r "s/(innodb_buffer_pool_size).*/\1 = ${pool_size}M #innodb_buffer_pool_size_inited/" /opt/zbox/etc/mysql/my.cnf
  fi

}
