#!/bin/bash

# FrankenPHP Go 启动脚本

SCRIPT_DIR="$(cd "$(dirname "$0")" && pwd)"
MAIN_BINARY="$SCRIPT_DIR/main"
PHP_DIR="$SCRIPT_DIR/php"

# 检查必要文件是否存在
check_files() {
    if [ ! -f "$MAIN_BINARY" ]; then
        echo "错误: 主程序不存在: $MAIN_BINARY"
        exit 1
    fi

    if [ ! -x "$MAIN_BINARY" ]; then
        echo "错误: 主程序不可执行: $MAIN_BINARY"
        exit 1
    fi

    if [ ! -d "$PHP_DIR" ]; then
        echo "错误: PHP 目录不存在: $PHP_DIR"
        exit 1
    fi

    if [ ! -d "$PHP_DIR/lib" ]; then
        echo "错误: 库目录不存在: $PHP_DIR/lib"
        exit 1
    fi
}

# 处理 php.ini 中的占位符
process_php_ini() {
    local php_ini="$SCRIPT_DIR/php.ini"

    if [ ! -f "$php_ini" ]; then
        return 0
    fi

    # 检查是否存在占位符
    if grep -q '%IONCUBE_PATH%\|%SESSION_PATH%' "$php_ini" 2>/dev/null; then
        echo "处理 php.ini 中的占位符..."

        # 创建临时文件
        local tmp_ini="${php_ini}.tmp"
        cp "$php_ini" "$tmp_ini"

        # 获取当前运行目录的绝对路径
        local run_dir="$SCRIPT_DIR"
        local ioncube_path="$run_dir/php"

        # 从 .env 文件读取 SESSION_PATH，如果没有则使用默认值
        local session_path="$run_dir/zentao/tmp/session"
        if [ -f "$SCRIPT_DIR/.env" ]; then
            local env_session=$(grep '^SESSION_PATH=' "$SCRIPT_DIR/.env" 2>/dev/null | cut -d'=' -f2)
            if [ -n "$env_session" ]; then
                session_path="$env_session"
            fi
        fi

        # 替换占位符
        sed -e "s|%IONCUBE_PATH%|$ioncube_path|g" \
            -e "s|%SESSION_PATH%|$session_path|g" \
            "$tmp_ini" > "$php_ini"

        rm -f "$tmp_ini"
        echo "已替换占位符: IONCUBE_PATH=$ioncube_path, SESSION_PATH=$session_path"
    fi
}

# 启动主程序
start_main() {
    check_files

    # 处理 php.ini 中的占位符
    process_php_ini

    # 设置库路径
    export LD_LIBRARY_PATH="$PHP_DIR/lib:$LD_LIBRARY_PATH"

    # 启动主程序
    echo "启动 FrankenPHP Go 服务..."
    echo "库路径: $PHP_DIR/lib"
    echo "主程序: $MAIN_BINARY"
    echo ""

    exec "$MAIN_BINARY" "$@"
}

# 主逻辑
start_main "$@"
