<?php
/**
 * Tencent Push Notification Service lib.
 *
 * @copyright Copyright 2009-2020 QingDao Nature Easy Soft Network Technology Co,LTD (www.cnezsoft.com)
 * @author    Wenrui LI <liwenrui@cnezsoft.com>
 * @package   tpns
 * @license   ZOSL (http://zpl.pub/page/zoslv1.html)
 * @version   $Id$
 * @Link      http://xuan.im
 */
class tpns
{
    /**
     * TPNS Endpoints.
     */
    const ENDPOINT_PUSH = 'https://api.tpns.tencent.com/v3/push/app';

    /**
     * Set accessId and secretKey.
     *
     * @param string $accessId
     * @param string $secretKey
     */
    public function __construct($accessId, $secretKey)
    {
        assert(isset($accessId) && isset($secretKey));

        $this->accessId  = $accessId;
        $this->secretKey = $secretKey;
    }

    /**
     * Execute curl, POST only.
     *
     * @param  string      $url     url of API endpoint.
     * @param  array       $params  POST params.
     * @param  string      $header  auth header, or any other headers.
     * @access private
     * @return object|bool returns false if failed to curl.
     */
    private function curl($url, $params = array(), $header = '')
    {
        $params = is_array($params) ? json_encode($params) : $params;

        $curlConfig = array(
            CURLOPT_URL            => $url,
            CURLOPT_HEADER         => false,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_CONNECTTIMEOUT => 3,
            CURLOPT_POST           => true,
            CURLOPT_POSTFIELDS     => $params,
            CURLOPT_HTTPHEADER     => $header
        );

        $ch = curl_init();
        curl_setopt_array($ch, $curlConfig);
        $data = curl_exec($ch);
        curl_close($ch);

        if(!empty($data))
        {
            $data = stripslashes($data);
            $data = json_decode($data);
        }
        if(json_last_error() !== JSON_ERROR_NONE) return false;
        return $data;
    }

    /**
     * Sign and send push request to TPNS.
     *
     * @param  object      $params
     * @access private
     * @return object|bool
     */
    public function push($params)
    {
        $authHeader = array('Authorization: Basic ' . base64_encode($this->accessId . ':' . $this->secretKey));
        if(is_object($params)) $params = (array)$params;
        return $this->curl(tpns::ENDPOINT_PUSH, $params, $authHeader);
    }

    /**
     * Make a tpns push pack with message and tokens.
     *
     * @param  object $message
     * @param  array  $tokenList
     * @access public
     * @return object
     */
    public function makePushPack($message, $tokenList)
    {
        $push = new stdclass();
        $push->audience_type = 'token_list';
        $push->token_list    = $tokenList;
        $push->message       = $message;
        $push->message_type  = 'notify';

        return $push;
    }

    /**
     * Make message for a push pack.
     *
     * @param  string $title
     * @param  string $content
     * @param  string $intent
     * @access public
     * @return object
     */
    public function makeMessagePack($title, $content = ' ', $intent = '')
    {
        $message = new stdclass();
        $message->title   = $title;
        $message->content = $content;
        /* Set android pack to message if intent presents. */
        if(!empty($intent)) $message->android = $this->makeAndroidPack($intent);

        return $message;
    }

    /**
     * Make the android part of a message pack.
     *
     * @param  string  $intent
     * @access private
     * @return object
     */
    private function makeAndroidPack($intent)
    {
        $android = new stdclass();
        $android->action = new stdclass();
        $android->action->action_type = 3; // 1 = open app, 2 = open browser, 3 = open intent.
        $android->action->intent = $intent;

        return $android;
    }
}
